#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
🌍 STROMKOSTEN-ANALYZER - ALL-IN-ONE VERSION
Eine einzige Datei für Windows, Mac UND Linux!

© 2025 Raymond Demitrio Dr. Tel - TEL Portal System
Notariell geschützt ⚖️

VERWENDUNG:
  Windows:  python Stromkosten-Analyzer-ALL-IN-ONE.py
  Mac:      python3 Stromkosten-Analyzer-ALL-IN-ONE.py
  Linux:    python3 Stromkosten-Analyzer-ALL-IN-ONE.py

FEATURES:
  ✅ Eine einzige Datei
  ✅ Funktioniert auf Windows, Mac, Linux
  ✅ Backend + Frontend + Datenbank ALLES DRIN
  ✅ Auto-Installation von Dependencies
  ✅ Auto-Start im Browser
  ✅ Keine externe Dateien benötigt!
"""

import sys
import os
import platform
import subprocess
import threading
import webbrowser
import time
import json
from pathlib import Path

# ============================================================================
# DEPENDENCY CHECK & AUTO-INSTALL
# ============================================================================

def check_and_install_dependencies():
    """Prüfe und installiere Dependencies automatisch"""
    required_packages = {
        'flask': 'Flask',
        'flask_cors': 'Flask-CORS',
    }
    
    missing = []
    for module, package in required_packages.items():
        try:
            __import__(module)
        except ImportError:
            missing.append(package)
    
    if missing:
        print(f"\n📦 Installiere fehlende Dependencies: {', '.join(missing)}")
        print("⏳ Das dauert etwa 1-2 Minuten...")
        
        try:
            subprocess.check_call([
                sys.executable, '-m', 'pip', 'install', '--user', '--quiet'
            ] + missing)
            print("✅ Dependencies installiert!\n")
            return True
        except:
            print("\n⚠️ Auto-Installation fehlgeschlagen.")
            print(f"💡 Bitte manuell installieren: pip install {' '.join(missing)}\n")
            return False
    
    return True

# Installiere Dependencies beim Import
if not check_and_install_dependencies():
    print("❌ Dependencies werden benötigt!")
    input("Drücke ENTER zum Beenden...")
    sys.exit(1)

# Jetzt können wir importieren
from flask import Flask, jsonify, request, send_from_directory
from flask_cors import CORS
from dataclasses import dataclass
from typing import Dict, List, Optional
from datetime import datetime
import hashlib

# ============================================================================
# CALCULATOR (Backend Logic)
# ============================================================================

@dataclass
class Geraet:
    """Gerät-Datenklasse"""
    name: str
    watt: float
    stunden_pro_tag: float
    kategorie: str = "Sonstiges"

@dataclass
class StromkostenErgebnis:
    """Ergebnis-Datenklasse"""
    geraet: Geraet
    kosten_pro_sekunde: float
    kosten_pro_stunde: float
    kosten_pro_tag: float
    kosten_pro_monat: float
    kosten_pro_jahr: float
    kosten_10jahre: float
    kwh_pro_jahr: float
    notariell_hash: str
    erstellt: str

class StromkostenCalculator:
    """Stromkosten Calculator"""
    
    def __init__(self, strompreis_kwh: float = 0.30):
        self.strompreis_kwh = strompreis_kwh
    
    def kosten(self, watt: float, stunden_pro_tag: float, preis_kwh: float) -> Dict:
        """Einfache Kostenberechnung"""
        kwh_pro_tag = (watt / 1000) * stunden_pro_tag
        kwh_pro_jahr = kwh_pro_tag * 365
        
        kosten_pro_stunde = (watt / 1000) * preis_kwh
        kosten_pro_tag = kwh_pro_tag * preis_kwh
        kosten_pro_monat = kosten_pro_tag * 30
        kosten_pro_jahr = kosten_pro_tag * 365
        
        return {
            'sekunde': kosten_pro_stunde / 3600,
            'stunde': kosten_pro_stunde,
            'tag': kosten_pro_tag,
            'monat': kosten_pro_monat,
            'jahr': kosten_pro_jahr,
            'kwh_jahr': kwh_pro_jahr
        }
    
    def berechne_kosten(self, geraet: Geraet, laufzeit_jahre: int = 10) -> StromkostenErgebnis:
        """Vollständige Kostenberechnung"""
        result = self.kosten(geraet.watt, geraet.stunden_pro_tag, self.strompreis_kwh)
        
        # Notarieller Hash
        hash_data = f"{geraet.name}_{geraet.watt}_{datetime.now().isoformat()}"
        notariell_hash = hashlib.sha256(hash_data.encode()).hexdigest()[:16]
        
        return StromkostenErgebnis(
            geraet=geraet,
            kosten_pro_sekunde=result['sekunde'],
            kosten_pro_stunde=result['stunde'],
            kosten_pro_tag=result['tag'],
            kosten_pro_monat=result['monat'],
            kosten_pro_jahr=result['jahr'],
            kosten_10jahre=result['jahr'] * laufzeit_jahre,
            kwh_pro_jahr=result['kwh_jahr'],
            notariell_hash=notariell_hash,
            erstellt=datetime.now().isoformat()
        )

# ============================================================================
# GERÄTE-DATENBANK (Inline)
# ============================================================================

GERAETE_DATENBANK = [
    {"name": "Kühlschrank Alt", "watt": 150, "stunden_pro_tag": 24, "kategorie": "Küche"},
    {"name": "Kühlschrank Neu (A+++)", "watt": 86, "stunden_pro_tag": 24, "kategorie": "Küche"},
    {"name": "Gefrierschrank", "watt": 200, "stunden_pro_tag": 24, "kategorie": "Küche"},
    {"name": "Waschmaschine", "watt": 2000, "stunden_pro_tag": 1, "kategorie": "Haushaltsgeräte"},
    {"name": "Trockner", "watt": 2500, "stunden_pro_tag": 1, "kategorie": "Haushaltsgeräte"},
    {"name": "Geschirrspüler", "watt": 1800, "stunden_pro_tag": 1, "kategorie": "Küche"},
    {"name": "LED TV 55\"", "watt": 100, "stunden_pro_tag": 5, "kategorie": "Unterhaltung"},
    {"name": "Laptop", "watt": 60, "stunden_pro_tag": 8, "kategorie": "Büro"},
    {"name": "Desktop PC", "watt": 300, "stunden_pro_tag": 8, "kategorie": "Büro"},
    {"name": "LED Lampe", "watt": 10, "stunden_pro_tag": 5, "kategorie": "Beleuchtung"},
]

# ============================================================================
# FRONTEND HTML (Inline)
# ============================================================================

HTML_TEMPLATE = """<!DOCTYPE html>
<html lang="de">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Stromkosten-Analyzer</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            min-height: 100vh;
            padding: 20px;
        }
        .container {
            max-width: 900px;
            margin: 0 auto;
            background: white;
            border-radius: 20px;
            padding: 40px;
            box-shadow: 0 20px 60px rgba(0,0,0,0.3);
        }
        h1 {
            color: #667eea;
            text-align: center;
            margin-bottom: 10px;
            font-size: 2.5em;
        }
        .subtitle {
            text-align: center;
            color: #666;
            margin-bottom: 30px;
        }
        .form-group {
            margin-bottom: 20px;
        }
        label {
            display: block;
            margin-bottom: 8px;
            color: #333;
            font-weight: 600;
        }
        input, select {
            width: 100%;
            padding: 12px;
            border: 2px solid #e0e0e0;
            border-radius: 8px;
            font-size: 16px;
            transition: border-color 0.3s;
        }
        input:focus, select:focus {
            outline: none;
            border-color: #667eea;
        }
        button {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 18px;
            font-weight: 600;
            cursor: pointer;
            transition: transform 0.2s;
        }
        button:hover {
            transform: translateY(-2px);
        }
        .result {
            margin-top: 30px;
            padding: 30px;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            border-radius: 15px;
            display: none;
        }
        .result.show {
            display: block;
            animation: slideIn 0.5s;
        }
        @keyframes slideIn {
            from {
                opacity: 0;
                transform: translateY(20px);
            }
            to {
                opacity: 1;
                transform: translateY(0);
            }
        }
        .result-item {
            display: flex;
            justify-content: space-between;
            padding: 12px 0;
            border-bottom: 1px solid rgba(0,0,0,0.1);
        }
        .result-item:last-child {
            border-bottom: none;
        }
        .result-label {
            font-weight: 600;
            color: #333;
        }
        .result-value {
            color: #667eea;
            font-weight: 700;
        }
        .footer {
            margin-top: 30px;
            text-align: center;
            color: #999;
            font-size: 14px;
        }
        .badge {
            display: inline-block;
            padding: 4px 12px;
            background: #667eea;
            color: white;
            border-radius: 20px;
            font-size: 12px;
            margin-left: 10px;
        }
    </style>
</head>
<body>
    <div class="container">
        <h1>⚡ Stromkosten-Analyzer</h1>
        <div class="subtitle">
            © 2025 Raymond Demitrio Dr. Tel - TEL Portal System
            <span class="badge">ALL-IN-ONE</span>
        </div>
        
        <div class="form-group">
            <label for="geraet">Gerät auswählen oder manuell eingeben:</label>
            <select id="geraet" onchange="loadGeraet()">
                <option value="">-- Manuell eingeben --</option>
            </select>
        </div>
        
        <div class="form-group">
            <label for="name">Gerätename:</label>
            <input type="text" id="name" placeholder="z.B. Kühlschrank">
        </div>
        
        <div class="form-group">
            <label for="watt">Leistung (Watt):</label>
            <input type="number" id="watt" placeholder="z.B. 150">
        </div>
        
        <div class="form-group">
            <label for="stunden">Stunden pro Tag:</label>
            <input type="number" id="stunden" placeholder="z.B. 24" step="0.1">
        </div>
        
        <div class="form-group">
            <label for="preis">Strompreis (€/kWh):</label>
            <input type="number" id="preis" value="0.30" step="0.01">
        </div>
        
        <button onclick="berechnen()">🔍 Berechnen</button>
        
        <div class="result" id="result">
            <h3 style="margin-bottom: 20px; color: #667eea;">📊 Ergebnis:</h3>
            <div class="result-item">
                <span class="result-label">Gerät:</span>
                <span class="result-value" id="res-name"></span>
            </div>
            <div class="result-item">
                <span class="result-label">Pro Tag:</span>
                <span class="result-value" id="res-tag"></span>
            </div>
            <div class="result-item">
                <span class="result-label">Pro Monat:</span>
                <span class="result-value" id="res-monat"></span>
            </div>
            <div class="result-item">
                <span class="result-label">Pro Jahr:</span>
                <span class="result-value" id="res-jahr"></span>
            </div>
            <div class="result-item">
                <span class="result-label">10 Jahre:</span>
                <span class="result-value" id="res-10jahre"></span>
            </div>
            <div class="result-item">
                <span class="result-label">Verbrauch/Jahr:</span>
                <span class="result-value" id="res-kwh"></span>
            </div>
        </div>
        
        <div class="footer">
            <p>🌍 Funktioniert auf Windows, Mac & Linux</p>
            <p>⚖️ Notariell geschützt - TEL Portal System</p>
        </div>
    </div>
    
    <script>
        let geraeteDB = [];
        
        // Lade Geräte-Datenbank
        async function ladeGeraete() {
            try {
                const response = await fetch('/api/datenbank');
                geraeteDB = await response.json();
                
                const select = document.getElementById('geraet');
                geraeteDB.forEach((g, i) => {
                    const option = document.createElement('option');
                    option.value = i;
                    option.textContent = `${g.name} (${g.watt}W)`;
                    select.appendChild(option);
                });
            } catch (err) {
                console.error('Fehler beim Laden der Datenbank:', err);
            }
        }
        
        // Lade Gerät aus Datenbank
        function loadGeraet() {
            const select = document.getElementById('geraet');
            const index = select.value;
            
            if (index !== '') {
                const g = geraeteDB[index];
                document.getElementById('name').value = g.name;
                document.getElementById('watt').value = g.watt;
                document.getElementById('stunden').value = g.stunden_pro_tag;
            }
        }
        
        // Berechnung durchführen
        async function berechnen() {
            const name = document.getElementById('name').value;
            const watt = parseFloat(document.getElementById('watt').value);
            const stunden = parseFloat(document.getElementById('stunden').value);
            const preis = parseFloat(document.getElementById('preis').value);
            
            if (!name || !watt || !stunden || !preis) {
                alert('Bitte alle Felder ausfüllen!');
                return;
            }
            
            try {
                const response = await fetch('/api/berechnen', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({
                        name, watt, stunden_pro_tag: stunden,
                        strompreis_kwh: preis, kategorie: 'Custom'
                    })
                });
                
                const data = await response.json();
                
                // Zeige Ergebnis
                document.getElementById('res-name').textContent = data.geraet.name;
                document.getElementById('res-tag').textContent = `€${data.kosten_pro_tag.toFixed(2)}`;
                document.getElementById('res-monat').textContent = `€${data.kosten_pro_monat.toFixed(2)}`;
                document.getElementById('res-jahr').textContent = `€${data.kosten_pro_jahr.toFixed(2)}`;
                document.getElementById('res-10jahre').textContent = `€${data.kosten_10jahre.toFixed(2)}`;
                document.getElementById('res-kwh').textContent = `${data.kwh_pro_jahr.toFixed(2)} kWh`;
                
                document.getElementById('result').classList.add('show');
            } catch (err) {
                alert('Fehler bei der Berechnung: ' + err.message);
            }
        }
        
        // Initial laden
        ladeGeraete();
    </script>
</body>
</html>
"""

# ============================================================================
# FLASK APP (Backend Server)
# ============================================================================

app = Flask(__name__)
CORS(app)

calculator = StromkostenCalculator()

@app.route('/')
def index():
    """Serve Frontend"""
    return HTML_TEMPLATE

@app.route('/api/health')
def health():
    """Health Check"""
    return jsonify({
        'status': 'healthy',
        'version': '1.0.0',
        'plattform': platform.system(),
        'python': sys.version.split()[0]
    })

@app.route('/api/datenbank')
def datenbank():
    """Geräte-Datenbank"""
    return jsonify(GERAETE_DATENBANK)

@app.route('/api/berechnen', methods=['POST'])
def berechnen():
    """Berechnung durchführen"""
    try:
        data = request.json
        
        geraet = Geraet(
            name=data['name'],
            watt=float(data['watt']),
            stunden_pro_tag=float(data['stunden_pro_tag']),
            kategorie=data.get('kategorie', 'Custom')
        )
        
        calc = StromkostenCalculator(strompreis_kwh=float(data.get('strompreis_kwh', 0.30)))
        ergebnis = calc.berechne_kosten(geraet)
        
        return jsonify({
            'geraet': {
                'name': ergebnis.geraet.name,
                'watt': ergebnis.geraet.watt,
                'stunden_pro_tag': ergebnis.geraet.stunden_pro_tag,
                'kategorie': ergebnis.geraet.kategorie
            },
            'kosten_pro_sekunde': ergebnis.kosten_pro_sekunde,
            'kosten_pro_stunde': ergebnis.kosten_pro_stunde,
            'kosten_pro_tag': ergebnis.kosten_pro_tag,
            'kosten_pro_monat': ergebnis.kosten_pro_monat,
            'kosten_pro_jahr': ergebnis.kosten_pro_jahr,
            'kosten_10jahre': ergebnis.kosten_10jahre,
            'kwh_pro_jahr': ergebnis.kwh_pro_jahr,
            'notariell_hash': ergebnis.notariell_hash,
            'erstellt': ergebnis.erstellt
        })
    except Exception as e:
        return jsonify({'error': str(e)}), 400

# ============================================================================
# LAUNCHER (Main Application)
# ============================================================================

class AllInOneLauncher:
    """All-In-One Launcher"""
    
    def __init__(self):
        self.platform = platform.system()
        self.port = 5000
        self.backend_process = None
    
    def print_banner(self):
        """Zeige Banner"""
        print("=" * 70)
        print("⚡ STROMKOSTEN-ANALYZER - ALL-IN-ONE VERSION")
        print("=" * 70)
        print(f"🖥️  Plattform: {self.platform}")
        print(f"🐍 Python: {sys.version.split()[0]}")
        print("=" * 70)
        print()
    
    def start_backend(self):
        """Starte Backend in separatem Thread"""
        def run_server():
            app.run(host='0.0.0.0', port=self.port, debug=False, use_reloader=False)
        
        thread = threading.Thread(target=run_server, daemon=True)
        thread.start()
        print(f"✅ Backend gestartet auf Port {self.port}")
    
    def wait_for_backend(self, max_wait=5):
        """Warte bis Backend bereit ist"""
        import urllib.request
        
        print("⏳ Warte auf Backend...")
        for i in range(max_wait):
            try:
                response = urllib.request.urlopen(f"http://127.0.0.1:{self.port}/api/health", timeout=1)
                if response.getcode() == 200:
                    print("✅ Backend ist bereit!")
                    return True
            except:
                time.sleep(1)
        return True
    
    def open_browser(self):
        """Öffne Browser"""
        print("🌐 Öffne Browser...")
        time.sleep(1)
        webbrowser.open(f"http://127.0.0.1:{self.port}")
        print("✅ Browser geöffnet!")
    
    def show_instructions(self):
        """Zeige Anweisungen"""
        print()
        print("=" * 70)
        print("✅ ANWENDUNG LÄUFT!")
        print("=" * 70)
        print()
        print(f"🔗 URL: http://127.0.0.1:{self.port}")
        print()
        print("💡 Tipps:")
        print("   - Lasse dieses Fenster offen")
        print("   - Beende mit CTRL+C")
        print()
        print("=" * 70)
        print()
    
    def run(self):
        """Hauptfunktion"""
        try:
            self.print_banner()
            self.start_backend()
            self.wait_for_backend()
            self.open_browser()
            self.show_instructions()
            
            print("🔄 Anwendung läuft... (CTRL+C zum Beenden)\n")
            
            # Halte am Laufen
            while True:
                time.sleep(1)
        
        except KeyboardInterrupt:
            print("\n")
            print("=" * 70)
            print("👋 Beende Anwendung...")
            print("=" * 70)
            print("Danke für die Nutzung!")
            print("© 2025 Raymond Demitrio Dr. Tel")
            print("=" * 70)
        
        except Exception as e:
            print(f"\n❌ Fehler: {e}")
            input("\nDrücke ENTER zum Beenden...")

# ============================================================================
# MAIN ENTRY POINT
# ============================================================================

if __name__ == "__main__":
    launcher = AllInOneLauncher()
    launcher.run()

